# Sage Theme Installation & Development Guide

Complete guide for installing, configuring, and developing with Sage WordPress theme.

---

## Table of Contents

1. [Prerequisites](#prerequisites)
2. [Installation](#installation)
3. [Development Workflow](#development-workflow)
4. [Cache Management](#cache-management)
5. [Common Issues & Solutions](#common-issues--solutions)
6. [Useful Commands Reference](#useful-commands-reference)

---

## Prerequisites

Before installing Sage, ensure you have the following installed:

### Required Software

- **PHP** >= 8.0
- **Composer** >= 2.0
- **Node.js** >= 18.0
- **npm** or **Yarn**
- **WordPress** >= 6.0

### Check Versions

```bash
# Check PHP version
php -v

# Check Composer version
composer -V

# Check Node.js version
node -v

# Check npm version
npm -v
```

---

## Installation

### Step 1: Create a New Sage Theme

Navigate to your WordPress themes directory:

```bash
cd wp-content/themes/
```

Create a new Sage theme using Composer:

```bash
composer create-project roots/sage your-theme-name
```

Or clone from GitHub:

```bash
git clone https://github.com/roots/sage.git your-theme-name
cd your-theme-name
```

### Step 2: Install Dependencies

```bash
# Install PHP dependencies
composer install

# Install Node.js dependencies
npm install

```

### Step 3: Build Assets

Go to vite.config.js and change the base url to your local or server url.

# example: base: '/wp-content/themes/your-theme-name/public/build/',

``

### Step 4: Configure Theme

Update `style.css` with your theme information:

```css
/*
Theme Name:         Your Theme Name
Theme URI:          https://yourwebsite.com
Description:        Your theme description
Version:            1.0.0
Author:             Your Name
Author URI:         https://yourwebsite.com
Text Domain:        your-theme-name
*/
```

### Step 5: Build theme

```bash
# Development build
npm run dev

# Production build
npm run build

# Watch for changes (auto-rebuild)
npm run dev -- --watch
```

### Step 5: Activate Theme

1. Go to WordPress Admin: `Appearance > Themes`
2. Activate your new Sage theme
3. Configure theme settings if needed

---

### Running Development Server

Start the development server with hot module replacement:

```bash
npm run dev
```

### Watch Mode (Recommended for Development)

Automatically rebuild on file changes:

```bash
npm run dev -- --watch
```

### Production Build

Create optimized production assets:

```bash
npm run build
```

### Linting & Code Quality

```bash
# Lint JavaScript
npm run lint:js

# Lint CSS/SCSS
npm run lint:css

# Fix linting issues automatically
npm run lint:js -- --fix
npm run lint:css -- --fix
```

---

## Cache Management

### Blade View Cache

Sage uses Laravel's Blade templating engine. Views are compiled and cached.

#### Clear Blade Cache

**Using WP-CLI (Recommended):**

```bash
# Clear all view cache
wp acorn view:clear

# Clear all cache (views, config, routes, etc.)
wp acorn optimize:clear
```

**Manual Method:**

Delete cached view files:

```bash
# Navigate to cache directory
cd wp-content/cache/acorn/framework/views/

```

#### Clear Specific Caches

```bash
# Clear config cache
wp acorn config:clear

# Clear route cache
wp acorn route:clear

# Clear event cache
wp acorn event:clear
```

### WordPress Object Cache

If using object caching (Redis, Memcached):

```bash
# Using WP-CLI
wp cache flush

# Or through WordPress Admin
# Tools > Site Health > Clear Cache (if available)
```

### Optimize for Production

Cache everything for production:

```bash
# Cache config, routes, views
wp acorn optimize

# Individual caching
wp acorn config:cache
wp acorn route:cache
wp acorn view:cache
```

---

## Common Issues & Solutions

### Issue of blade syntax: "Unable to locate a class or view for component"

**Error:**

```
InvalidArgumentException: Unable to locate a class or view for component [component-name]
```

**Solution:**

1. Check component file path matches usage
2. Clear view cache: `wp acorn view:clear`
3. Ensure proper naming (use dots for subdirectories: `<x-folder.component>`)

---

### General Issue : Blade Syntax Errors

**Solution:**

1. Check for unclosed tags: `@if`, `@foreach`, etc.
2. Ensure proper Blade directive syntax
3. Clear view cache: `wp acorn view:clear`
4. Check error logs: `wp-content/debug.log`

---

## Useful Commands Reference

### Acorn (Laravel) Commands

```bash
# List all available commands
wp acorn list

# Clear all caches
wp acorn optimize:clear

# Clear specific caches
wp acorn view:clear
wp acorn config:clear
wp acorn route:clear
wp acorn event:clear

# Cache for production
wp acorn optimize
wp acorn config:cache
wp acorn route:cache
wp acorn view:cache

# Run artisan commands
wp acorn make:component ComponentName
wp acorn make:view view-name

# Check Acorn version
wp acorn --version
```

### NPM/Build Commands

```bash
# Install dependencies
npm install

# Development build
npm run dev

# Production build
npm run build

# Watch mode
npm run dev -- --watch

# Clean build directory
npm run clean

# Lint code
npm run lint
npm run lint:js
npm run lint:css

# Update dependencies
npm update
```

### Composer Commands

```bash
# Install dependencies
composer install

# Update dependencies
composer update

# Update specific package
composer update vendor/package

# Dump autoload
composer dump-autoload

# Validate composer.json
composer validate
```

### Database & WordPress

```bash
# Flush WordPress rewrite rules
wp rewrite flush

# Regenerate .htaccess
wp rewrite flush --hard

# Export/Import database
wp db export backup.sql
wp db import backup.sql

# Search and replace URLs
wp search-replace 'old-url.com' 'new-url.com'
```

---

## Development Best Practices

### 1. Always Use Version Control

```bash
git init
git add .
git commit -m "Initial commit"
```

### 2. Create .gitignore

```gitignore
# Dependencies
/node_modules
/vendor

# Build files
/public

# Cache
/storage
/wp-content/cache

# Environment
.env
.env.*

# OS files
.DS_Store
Thumbs.db
```

### 3. Use Environment Variables

Create `.env` file (don't commit to git):

```env
WP_ENV=development
WP_HOME=http://localhost:3000
WP_SITEURL=${WP_HOME}/wp
```

### 4. Regular Cache Clearing During Development

Add this to your workflow:

```bash
# After making Blade changes
wp acorn view:clear && npm run dev
```

## Resources

- **Sage Documentation**: https://roots.io/sage/docs
- **Acorn Documentation**: https://roots.io/acorn/docs
- **Blade Templates**: https://laravel.com/docs/blade
- **Bud.js (Build Tool)**: https://bud.js.org
- **Roots Discourse**: https://discourse.roots.io

---

## Quick Troubleshooting Checklist

When something isn't working:

1. ✅ Clear view cache: `wp acorn view:clear`
2. ✅ Rebuild assets: `npm run build`
3. ✅ Clear browser cache: `Ctrl + Shift + R`
4. ✅ Check error logs: `wp-content/debug.log`
5. ✅ Reinstall dependencies: `rm -rf node_modules && npm install`
6. ✅ Restart development server
7. ✅ Check file permissions
8. ✅ Verify WordPress and PHP versions meet requirements

---

**Last Updated**: December 2024  
**Sage Version**: 10.x  
**WordPress**: 6.x+
